import { observable, action, computed } from 'mobx'
import {
  mutateValueByPath,
  getValueByPath,
  deepClone,
  getNumber
} from 'layersWp/utils/helper'
import snapshot from './snapshot'
import uuidv4 from 'uuid/v4'
import { sceneGen, layerGen } from 'layersWp/constructors/other-default'
import HookStore from './hook'
import * as types from 'layersWp/utils/itemType'
class SceneStore {
  @observable
  layerList = []
  @observable
  editingLayer = {
    id: ''
  }

  @observable
  editingTarget = {
    id: ''
  }

  @observable
  scene = sceneGen()
  @observable
  skinDesc = ''
  @observable
  skinType = 0
  @observable
  skinTags = []
  // 重置数据
  @action
  reset = () => {
    let layer = layerGen('主场景', true)
    this.editingTarget = { id: '' }
    this.layerList = [layer]
    this.editingLayer = { id: layer.layerId }
    this.skinTags = []
    this.scene = sceneGen()
    snapshot.addDep({
      sceneSnapshot: {
        target: this,
        keys: ['scene', 'itemList', 'editingTarget']
      }
    })
  }

  // 初始化数据
  @action
  init = (page = null, layers = []) => {
    // 默认显示第一个场景
    this.chooseLayer({ layerId: layers[0].layerId })
    this.layerList = layers
    if (page) {
      this.scene = page
      HookStore.walkHooks('onSceneInnerHeightChangeEnd', {
        height: getNumber(page.height)
      })
      HookStore.walkHooks('onSceneInnerWidthChangeEnd', {
        width: getNumber(page.width)
      })
    }
  }

  // 改变舞台属性
  @action
  changeSceneProps = obj => {
    // 支持.操作符修改属性
    Object.keys(obj).forEach(key => {
      const path = key
      const value = obj[key]
      mutateValueByPath(this.scene, path, value)
    })
    snapshot.addSnapshot()
  }

  // 向场景中增加部件
  @action
  addItem = item => {
    let layer = this.currentLayer
    // 处理zIndex
    const maxIndex = layer.itemList
      .filter(item => {
        return item.widgetType !== types.PopupType
      })
      .reduceRight((acc, item) => {
        const zIndex = item[item.widgetType].style.zIndex
        if (zIndex > acc) {
          acc = zIndex
        }
        return acc
      }, 0)
    if (maxIndex > 0) {
      item[item.widgetType].style.zIndex = maxIndex
    }
    layer.itemList.push(item)
    this.chooseTarget({ targetId: item.targetId })
  }

  // 复制场景部件
  @action
  copyAndPasteItem = targetId => {
    let layer = this.currentLayer
    let item = layer.itemList.find(item => item.targetId === targetId)
    if (item) {
      let newItem = deepClone(item)
      // 重新更改id
      newItem.targetId = `${newItem.widgetType}_${uuidv4()}`
      newItem[newItem.widgetType].id = newItem.targetId
      // 重命名
      newItem[newItem.widgetType].name += '-copy'
      // 重新设置位置
      newItem[newItem.widgetType].style.top =
        getNumber(newItem[newItem.widgetType].style.top) + 20 + 'px'
      newItem[newItem.widgetType].style.left =
        getNumber(newItem[newItem.widgetType].style.left) + 20 + 'px'
      this.addItem(newItem)
    }
  }

  // 删除当前选中item
  @action
  removeCurrItem = () => {
    let layer = this.currentLayer
    const index = layer.itemList.findIndex(
      item => item.targetId === this.editingTarget.id
    )
    if (index !== -1) {
      this.editingTarget.id = ''
      layer.itemList.splice(index, 1)
    }
    snapshot.addSnapshot()
  }
  // 选中编辑弹层
  @action
  chooseLayer = ({ layerId }) => {
    this.editingLayer.id = layerId
    snapshot.addSnapshot()
  }
  // 选中编辑部件
  @action
  chooseTarget = ({ targetId }) => {
    this.editingTarget.id = targetId
    snapshot.addSnapshot()
  }

  // !请勿直接使用此方法，使用changeItemProps代替
  @action
  changeItemProp = (path, value) => {
    let layer = this.currentLayer
    if (!this.editingTarget.id) return false
    const target = layer.itemList.find(
      item => item.targetId === this.editingTarget.id
    )
    if (!target) return false
    mutateValueByPath(target[target.widgetType], path, value)
  }

  // 改变当前正在编辑的部件多个属性
  @action
  changeItemProps = obj => {
    Object.keys(obj).forEach(key => {
      this.changeItemProp(key, obj[key])
    })
    snapshot.addSnapshot()
  }

  // 根据targetId操作数组项
  @action
  mutateArrayProp = (targetId, path, arrayFuncName, ...params) => {
    let layer = this.currentLayer
    const items = [...layer.itemList]
    let target = items.find(item => item.targetId === targetId)

    if (!target) return
    let arr = getValueByPath(target[target.widgetType], path)

    if (!(arr instanceof Array)) return
    Array.prototype[arrayFuncName].call(arr, ...params)
  }

  // 根据targetId改变属性
  @action
  changeItemPropsById = (targetId, obj) => {
    let layer = this.currentLayer
    const items = [...layer.itemList]
    let target = items.find(item => item.targetId === targetId)
    if (!target) return
    Object.keys(obj).forEach(key => {
      mutateValueByPath(target[target.widgetType], key, obj[key])
    })
  }

  // 根据targetId删除元素
  @action
  removeItemById = targetId => {
    let layer = this.currentLayer
    const index = layer.itemList.findIndex(item => item.targetId === targetId)
    if (index !== -1) {
      this.editingTarget.id = ''
      layer.itemList.splice(index, 1)
    }
    snapshot.addSnapshot()
  }

  // 根据targetId查询元素属性（非响应式）
  @action
  getItemPropById = (targetId, path) => {
    let layer = this.currentLayer
    const items = [...layer.itemList]
    let target = items.find(item => item.targetId === targetId)
    if (!target) return '404'
    return getValueByPath(target[target.widgetType], path)
  }

  // 获取层级
  @action
  getZIndex = () => {
    let layer = this.currentLayer
    // 处理zIndex
    const items = layer.itemList.filter(item => {
      return item.widgetType !== types.PopupType
    })
    const maxZIndex = items.reduceRight((acc, item) => {
      const zIndex = item[item.widgetType].style.zIndex
      if (zIndex > acc) {
        acc = zIndex
      }
      return acc
    }, 0)
    const minZIndex = items.reduce((acc, item) => {
      const zIndex = item[item.widgetType].style.zIndex
      if (zIndex < acc) {
        acc = zIndex
      }
      return acc
    }, 100000)
    return {
      maxZIndex,
      minZIndex
    }
  }

  // 添加弹层
  @action
  addLayer = () => {
    let index = this.layerList.length + 1
    let layer = layerGen(`场景-${index}`)
    this.layerList.push(layer)
    this.chooseLayer({ layerId: layer.layerId })
  }
  // 删除弹层
  @action
  removeLayerById = () => {
    const index = this.layerList.findIndex(
      item => item.layerId === this.editingLayer.id
    )
    // 第一个为主场景，不可删除
    if (index === 0) {
      return index
    }
    if (index !== -1) {
      this.editingLayer.id = ''
      this.layerList.splice(index, 1)
    }
    return index
  }
  @action
  changeLayerProps = obj => {
    if (!this.currentLayer) return
    Object.keys(obj).forEach(key => {
      mutateValueByPath(this.currentLayer, key, obj[key])
    })
  }
  // 根据layerId改变属性
  @action
  changeLayerPropsById = (layerId, obj) => {
    let layer = this.layerList.find(item => item.layerId === layerId)
    if (!layer) return
    Object.keys(obj).forEach(key => {
      mutateValueByPath(layer, key, obj[key])
    })
  }
  // 设置描述
  @action
  setDesc = desc => {
    this.skinDesc = desc
  }
  // 设置类型
  @action
  setSkinType = type => {
    this.skinType = type
  }

  // 设置标签
  @action
  setSkinTags = tags => {
    this.skinTags = tags
  }

  // 当前场景
  // TODO 和editingLayer合并
  @computed get currentLayer() {
    return (
      this.layerList.find(layer => layer.layerId === this.editingLayer.id) ||
      this.layerList[0]
    )
  }

  /**
   * 当前编辑的Item
   * @return {Item | undefined}
   */
  @computed
  get currentItem() {
    return (this.currentLayer.itemList || []).find(
      item => item.targetId === this.editingTarget.id
    )
  }
}

const sceneStore = new SceneStore()

sceneStore.reset()

export default sceneStore
