import React from 'react';
import { hot } from 'react-hot-loader';
import { Layout, Input, Icon, Modal, Button, message, Select } from 'antd';
import ajax from 'libs/ajax';
import util from 'libs/util';
import PluginItemSimple from 'basics/components/pluginItemSimple/index';
import PluginAttrItem from 'basics/components/pluginAttrItem/index';
import { Sortable } from 'react-sortablejs';
import Styles from './index.less';
const { Option } = Select;
const { Header, Content } = Layout;
class ActivityList extends React.Component {
  constructor(props) {
    super(props);
    this.state = {
      layoutList: [],
      undo: [],
      redo: [],
      hasUndo: false,
      hasRedo: false,
      cateList: [],
      pluginList: {},
      editList: {},
      activity: {
        title: null,
        summary: null,
        components: []
      } // 整个活动相关配置项
    };
  }
  componentWillMount() {
    this.getCateList();
    this.getActivityDetail();
    this.getLayoutList();
  }

  getActivityDetail() {
    let { id } = this.props.router.location.query;
    if (!id) return;
    ajax({
      url: 'basics/activity/detail',
      params: {
        id
      }
    })
      .then(data => {
        this.stateSet(function(state) {
          state.activity = {
            id: data.id,
            title: data.title,
            layoutId: data.layoutId,
            summary: data.summary,
            components: data.components
          };
        });
      })
      .catch(err => err);
  }

  // 获取模板列表
  getLayoutList() {
    ajax({
      url: '/basics/layout',
      params: {
        pageSize: 100
      }
    }).then(res => {
      this.setState(state => {
        return {
          layoutList: res.lists
        };
      });
    });
  }

  // 获取分类
  getCateList() {
    ajax({
      url: 'basics/cate',
      params: {
        pageSize: 100
      }
    })
      .then(res => {
        this.setState({
          cateList: res.lists
        });
        res.lists &&
          res.lists.forEach(item => {
            this.getPluginList(item.id);
          });
      })
      .catch(err => err);
  }

  // 获取当前分类下的插件
  getPluginList(cateId) {
    const pluginList = this.state.pluginList;
    if (pluginList[cateId] && pluginList[cateId].loaded) {
      return false;
    }
    if (!pluginList[cateId]) {
      pluginList[cateId] = {
        data: [],
        loaded: false
      };
    }
    ajax({
      url: 'basics/component',
      params: {
        pageSize: 100,
        cateId
      }
    })
      .then(res => {
        pluginList[cateId].data = res.lists;
        pluginList[cateId].loaded = true;
        this.setState({
          pluginList
        });
      })
      .catch(err => err);
  }

  // 返回按钮
  goBack() {
    Modal.confirm({
      title: '确定离开当前页面?',
      onOk: () => {
        this.props.router.push('/activity');
      }
    });
  }

  // 预览区组件变动
  handleComponentsChange(items) {
    items = items.map(i => {
      i = JSON.parse(i);
      return {
        id: i.id,
        title: i.title,
        thumb: i.thumb,
        html: i.html,
        css: i.css,
        js: i.js,
        option: i.option || {},
        config: i.config || []
      };
    });

    this.setActivity(function(preState) {
      preState.activity.components = items;
      return preState;
    });
  }

  // 更改state
  stateSet(cb) {
    this.setState(function(preState) {
      cb(preState);
      return preState;
    });
  }

  // 更改活动相关
  setActivity(cb) {
    let activity;
    let undo;
    this.setState(preState => {
      activity = util.copy(preState.activity, {});
      preState.undo.push(activity);
      undo = JSON.parse(JSON.stringify(preState.undo));
      this.applyUndoRedo();
      cb(preState);
      return {
        activity: preState.activity,
        undo
      };
    });
  }

  // 删除指定index组件
  deleteComponent(index) {
    this.setActivity(function(preState) {
      preState.activity.components.splice(index, 1);
    });
  }

  // 展示可配置项
  showConfig(option) {
    let { config } = option;
    config = JSON.parse(config.replace(/\s/g, '').replace(/([a-zA-Z0-9]+?):/g, '"$1":'));
    this.setState(function(preState) {
      preState.editList = {
        config: config,
        title: option.title,
        key: option.key
      };
      return preState;
    });
  }

  // 组件配置项变动
  handleEditChange = (e, val) => {
    let value;
    let type = val.type;
    let { key } = this.state.editList;
    if (type === 'number' || type === 'boolean') {
      value = e;
    } else if (type === 'image') {
      let res = e.file.response;
      if (res && res.code === 0) {
        value = res.data;
      } else if (res && res.msg) {
        message.error(res.msg);
      }
    } else if (type === 'color') {
      value = e.hex;
    } else {
      value = e.target.value;
    }

    this.setActivity(function(preState) {
      preState.activity.components[key].option[val.key] = value;
    });
  };

  // 活动配置项变动
  handleEditChange2 = (e, key) => {
    let value;
    switch (key) {
      case 'layoutId':
        value = e;
        break;
      default:
        value = e.target.value;
    }
    this.setActivity(function(preState) {
      preState.activity[key] = value;
    });
  };

  // 保存活动
  activitySave = () => {
    let data = this.state.activity;
    if (!data.title) {
      message.error('活动标题不能为空');
      return null;
    }

    if (!data.layoutId) {
      message.error('活动模板不能为空');
      return null;
    }

    if (data.components.length <= 0) {
      message.error('活动组件不能为空');
      return null;
    }

    data.components = data.components.map(item => {
      return {
        id: item.id,
        option: item.option
      };
    });

    ajax({
      url: 'basics/activity',
      method: data.id ? 'put' : 'post',
      data
    })
      .then(res => {
        !data.id && this.props.router.push('/activity');
        message.success('活动保存成功');
      })
      .catch(err => err);
  };

  // 应用撤销前进
  applyUndoRedo() {
    this.stateSet(function(state) {
      state.hasUndo = state.undo.length > 0;
      state.hasRedo = state.redo.length > 0;
    });
  }

  // 前进
  activityRedo = () => {
    const activity = util.copy(this.state.activity, {});
    this.stateSet(function(state) {
      state.undo.push(activity);
      state.activity = state.redo.pop();
    });
    this.applyUndoRedo();
  };

  // 撤销
  activityUndo = () => {
    const activity = util.copy(this.state.activity, {});
    this.stateSet(function(state) {
      state.redo.push(activity);
      state.activity = state.undo.pop();
    });
    this.applyUndoRedo();
  };

  activityPreview = () => {
    window.open(`/api/basics/activity/preview/${this.state.activity.id}`);
  };

  activityPublish = () => {
    let id = this.state.activity.id;
    Modal.confirm({
      title: '确定发布该活动?',
      confirmLoading: true,
      okType: 'success',
      onOk: () => {
        return ajax({
          url: `/basics/activity/publish`,
          data: {
            id
          },
          method: 'post'
        })
          .then(data => {
            message.success('发布成功');
          })
          .catch(err => err);
      }
    });
  };

  // 渲染待选区插件项
  renderPluginItem(list) {
    return list.map(item => (
      <Sortable
        className={Styles.pluginItemContainer}
        options={{ group: { name: 'simple', pull: 'clone', put: false } }}
        key={item.id}
      >
        <div data-id={JSON.stringify(item)}>
          <PluginItemSimple item={item} />
        </div>
      </Sortable>
    ));
  }

  // 渲染待选区插件列表
  renderPluginList(item) {
    const { pluginList } = this.state;
    return (
      <div key={item.id} className={Styles.cateListContainer}>
        <span className={Styles.listTitle}>{item.title}</span>
        {pluginList[item.id] &&
          pluginList[item.id].data &&
          this.renderPluginItem(pluginList[item.id].data)}
      </div>
    );
  }

  // 渲染预览区组件列表
  renderComponentItem(val, key) {
    return (
      val && (
        <div
          key={key}
          data-id={JSON.stringify(val)}
          className={Styles.comItem}
          style={{ backgroundImage: 'url(' + val.thumb + ')' }}
        >
          <div
            className={Styles.comItemAction}
            onClick={this.showConfig.bind(this, { config: val.config, title: val.title, key })}
          >
            <Button type="danger" onClick={this.deleteComponent.bind(this, key)}>
              <Icon type="delete" />
            </Button>
          </div>
        </div>
      )
    );
  }

  // 渲染组件可配置项
  renderPluginEdit() {
    const { editList, activity } = this.state;
    const { config = [] } = editList;
    return config.map(item => {
      return (
        <div key={item.key} className={Styles.editItem}>
          <span className={Styles.editItemTitle}>{item.title}</span>
          <div>
            <PluginAttrItem
              item={item}
              option={activity.components[editList.key].option}
              onChange={this.handleEditChange}
            />
          </div>
        </div>
      );
    });
  }

  render() {
    const { layoutList, cateList = [], editList = {}, activity = {} } = this.state;
    return (
      <Layout className="layout">
        <Header className={Styles.header}>
          <span className={Styles.back} onClick={this.goBack.bind(this)}>
            <Icon type="arrow-left" />
            返回
          </span>
          <span
            className={Styles.action}
            onClick={this.activityUndo}
            disabled={!this.state.hasUndo}
          >
            <Icon type="left-circle-o" />
            撤销
          </span>
          <span
            className={Styles.action}
            onClick={this.activityRedo}
            disabled={!this.state.hasRedo}
          >
            <Icon type="right-circle-o" />
            前进
          </span>
          <div className={Styles.headerRight}>
            <span className={Styles.action} onClick={this.activitySave}>
              <Icon type="save" />
              保存
            </span>
            <span className={Styles.action} onClick={this.activityPreview}>
              <Icon type="eye-o" />
              预览
            </span>
            <span className={Styles.action} onClick={this.activityPublish}>
              <Icon type="sync" />
              发布
            </span>
          </div>
        </Header>
        <Content style={{ padding: '72px 50px 75px 50px' }}>
          <div className={Styles.left} id="pluginList">
            {cateList.map(item => this.renderPluginList(item))}
          </div>
          <div className={Styles.content}>
            <div className={Styles.preview}>
              <div className={Styles.previewContent} id="preview">
                <Sortable
                  options={{ group: { name: 'simple', pull: 'false' } }}
                  onChange={items => this.handleComponentsChange(items)}
                >
                  {activity.components &&
                    activity.components.map((val, key) => this.renderComponentItem(val, key))}
                </Sortable>
              </div>
            </div>
          </div>
          <div className={Styles.right}>
            <div className={Styles.rightTitle}>舞台</div>
            <div className={Styles.editItem}>
              <span className={Styles.editItemTitle}>标题</span>
              <Input onChange={e => this.handleEditChange2(e, 'title')} value={activity['title']} />
            </div>
            <div className={Styles.editItem}>
              <span className={Styles.editItemTitle}>模板</span>
              <Select
                onChange={e => this.handleEditChange2(e, 'layoutId')}
                value={activity['layoutId']}
              >
                {layoutList &&
                  layoutList.map(item => {
                    return (
                      <Option key={item.id} value={item.id}>
                        {item.title}
                      </Option>
                    );
                  })}
              </Select>
            </div>
            <div className={Styles.editItem}>
              <span className={Styles.editItemTitle}>简介</span>
              <Input
                onChange={e => this.handleEditChange2(e, 'summary')}
                value={activity['summary']}
              />
            </div>
            <div className={Styles.rightTitle}>{editList.title}</div>
            {this.renderPluginEdit()}
          </div>
        </Content>
      </Layout>
    );
  }
}

export default hot(module)(ActivityList);
