const webpack = require('webpack')
const merge = require('webpack-merge')
const TerserWebpackPlugin = require('terser-webpack-plugin')
const ProgressBarPlugin = require('progress-bar-webpack-plugin')
const MiniCssExtractPlugin = require('mini-css-extract-plugin')
const OptimizeCSSPlugin = require('optimize-css-assets-webpack-plugin')
const HtmlWebpackPlugin = require('html-webpack-plugin')
const HtmlWebpackHarddiskPlugin = require('html-webpack-harddisk-plugin')
const chalk = require('chalk')

const utils = require('./utils')

const config = require('./config')
const webpackBaseConfig = require('./webpack.base.conf')

const { sourceMap, cssSourceMap } = config.build

module.exports = merge(webpackBaseConfig, {
  output: {
    path: config.path,
    filename: '[name].[chunkhash].js',
    publicPath: config.build.publicPath,
    chunkFilename: '[chunkhash].js'
  },
  mode: 'production',
  devtool: sourceMap ? '#source-map' : false,
  module: {
    rules: [
      {
        test: /\.less$/,
        exclude: /node_modules|antd/,
        use: [
          {
            loader: MiniCssExtractPlugin.loader
          },
          {
            loader: 'css-loader',
            options: {
              modules: true,
              localIdentName: '[local]_[hash:base64:5]',
              sourceMap: cssSourceMap
            }
          },
          {
            loader: 'postcss-loader',
            options: {
              sourceMap: cssSourceMap
            }
          },
          {
            loader: 'less-loader',
            options: {
              javascriptEnabled: true,
              sourceMap: cssSourceMap
            }
          }
        ]
      },
      {
        test: /(\.less|\.css)$/,
        include: /node_modules|antd/,
        use: [
          {
            loader: MiniCssExtractPlugin.loader
          },
          {
            loader: 'css-loader',
            options: {
              sourceMap: cssSourceMap
            }
          },
          {
            loader: 'less-loader',
            options: {
              javascriptEnabled: true,
              sourceMap: cssSourceMap
            }
          }
        ]
      }
    ]
  },
  optimization: {
    minimizer: [
      new TerserWebpackPlugin({
        terserOptions: {
          compress: {
            drop_console: true
          }
        },
        cache: true,
        parallel: true,
        sourceMap
      })
    ],
    splitChunks: {
      chunks: 'all',
      automaticNameDelimiter: '-',
      cacheGroups: {
        styles: {
          name: 'styles',
          test: /\.(css|less)$/,
          chunks: 'all',
          enforce: true
        }
      }
    }
  },
  plugins: [
    new webpack.NoEmitOnErrorsPlugin(),
    new webpack.HashedModuleIdsPlugin(),
    new webpack.optimize.ModuleConcatenationPlugin(),
    new ProgressBarPlugin({
      format: chalk.yellow('打包中 [:bar] :current/:total :percent :elapseds :msg'),
      complete: '●',
      incomplete: '○',
      width: 20
    }),
    new MiniCssExtractPlugin({
      filename: '[name].[contenthash].css'
    }),
    new OptimizeCSSPlugin({
      cssProcessorOptions: {
        safe: true
      }
    }),
    new webpack.ContextReplacementPlugin(/moment[/\\]locale$/, /zh-cn/),
    ...utils
      .computeTemplate({
        minify: true
      })
      .map(el => {
        return new HtmlWebpackPlugin(el)
      }),
    new HtmlWebpackHarddiskPlugin()
  ]
})
